<?php
/*
  Plugin Name: Botoscope
  Plugin URI: https://botoscope.com/
  Description: Telegram eCommerce system
  Requires at least: WP 6.0
  Tested up to: WP 6.9
  Author: botoscope
  Author URI: https://botoscope.com/
  Version: 1.0.0
  Requires PHP: 8.3
  Tags: woocommerce, telegram, store, shop, bot
  Text Domain: botoscope
  Domain Path: /languages
  WC requires at least: 9.0
  WC tested up to: 10.5
  Requires Plugins: woocommerce
  Forum URI: https://botoscope.com/documentation
 */

//25-02-2026
if (!defined('ABSPATH')) {
    exit;
}

if (!defined('BOTOSCOPE_CLIENT_API_KEY')) {
    add_filter('plugin_row_meta', function ($links, $file) {
        if (strpos($file, 'botoscope/index.php') !== false) {
            $links[] = '<br><br><a href="https://botoscope.com/start/" target="_blank" style="color: #d54e21; font-weight: bold;">' . esc_html__('Read Setup Instructions', 'botoscope') . '</a>';
        }
        return $links;
    }, 10, 2);
}

include_once 'lib/helper.php';
include_once 'lib/storage.php';
include_once 'lib/hooks.php';
include_once 'classes/app.php';
include_once 'classes/extensions.php';
include_once 'classes/currency.php';
include_once 'classes/payment.php';
include_once 'classes/controls.php';
include_once 'classes/translations.php';
include_once 'classes/taxonomies.php';
include_once 'classes/users.php';
include_once 'rest/orders.php';
include_once 'rest/allrest.php';

define('BOTOSCOPE_PLUGIN_NAME', plugin_basename(__FILE__));
define('BOTOSCOPE_VERSION', '1.0.0'); //dev
//define('BOTOSCOPE_VERSION', uniqid('bs-')); //dev tests
define('BOTOSCOPE_PATH', plugin_dir_path(__FILE__));
define('BOTOSCOPE_LINK', plugin_dir_url(__FILE__));
define('BOTOSCOPE_ASSETS_LINK', BOTOSCOPE_LINK . 'assets/');
define('BOTOSCOPE_EXT_PATH', BOTOSCOPE_PATH . 'exts/');
define('BOTOSCOPE_EXT_LINK', BOTOSCOPE_LINK . 'exts/');
define('BOTOSCOPE_CUSTOM_EXT_PATH', WP_CONTENT_DIR . '/botoscope/exts/');
define('BOTOSCOPE_CUSTOM_EXT_LINK', WP_CONTENT_URL . '/botoscope/exts/');

add_action('before_woocommerce_init', function () {
    if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

add_action('plugins_loaded', function () {
    load_plugin_textdomain('botoscope', false, basename(dirname(__FILE__)) . '/languages');
});

class Botoscope {

    private $slug = 'botoscope';
    public $apps;
    public $storage;
    public $debug;
    //apps
    public $extensions;
    public $payment;
    public $options;
    public $controls;
    public $taxonomies;
    public $allrest;
    public $rest_orders;
    public $languages;
    public $translations;
    //exts
    public $sizecharts; //todo
    public $shipping;
    public $pickup_points; //todo
    public $coupons;
    public $support;
    public $elogios;
    public $advertising;
    public $product_attributes;
    public $interface_translations;
    public $currency;
    public $products;
    public $b2b;
    public $users;
    public $broadcast;
    public $booking;
    public $business_in_pocket;
    public $marketing;
    public $no_bot;
    public static $allowed_slugs = ['botoscope-filter', 'botoscope-product-details', 'botoscope-thank-you', 'botoscope-variation-gallery',
        'botoscope-media-casting', 'botoscope-refund-policy', 'botoscope-privacy-policy', 'botoscope-shipping-policy', 'botoscope-chat'];

    public function __construct() {

        register_activation_hook(__FILE__, function () {
            flush_rewrite_rules();
        });

        $this->debug = apply_filters('botoscope_debug', 0);
        $this->no_bot = !defined('BOTOSCOPE_BOT_TOKEN');
        $this->storage = new BOTOSCOPE_STORAGE();

        $this->allrest = new BOTOSCOPE_REST_ALLREST($this);
        $this->rest_orders = new BOTOSCOPE_REST_ORDERS($this);

        $this->apps = ['extensions', 'currency', 'payment', 'controls', 'taxonomies', 'users'];

        foreach ($this->apps as $app) {
            $class_name = 'BOTOSCOPE_' . strtoupper($app);
            if (class_exists($class_name)) {
                $this->$app = new $class_name(['botoscope' => $this]);
            }
        }

        $this->languages = include 'data/supported_languages.php';
        $this->translations = new BOTOSCOPE_TRANSLATIONS(['botoscope' => $this]); //this is service for all apps

        $this->set_ajax_actions();
        add_action('admin_enqueue_scripts', array($this, 'admin_enqueue_scripts'));
    }

    public function is_botoscope_page(): bool {
        if (!is_page()) {
            return false;
        }

        $obj = get_queried_object();
        $slug = is_object($obj) ? ($obj->post_name ?? '') : '';

        return $slug && in_array($slug, Botoscope::$allowed_slugs, true);
    }

    public function init() {
        //Intercepting rendering and outputting a "minimal" template
        add_action('template_redirect', function () {
            if (!$this->is_botoscope_page()) {
                return;
            }

            status_header(200);
            //nocache_headers();
            ?>
            <!DOCTYPE html>
            <html <?php language_attributes(); ?>>
                <head>
                    <meta charset="<?php bloginfo('charset'); ?>">
                    <meta name="viewport" content="width=device-width, initial-scale=1">
                    <meta name="robots" content="noindex,nofollow" />
                    <?php wp_enqueue_style('botoscope-page', apply_filters('botoscope_page_styles_link', BOTOSCOPE_ASSETS_LINK . 'css/page.css'), [], BOTOSCOPE_VERSION); ?>
                    <?php wp_head(); ?>
                </head>
                <body <?php body_class(); ?>>
                    <main>
                        <article>
                            <?php
                            while (have_posts()) {
                                the_post();
                                the_content();
                            }
                            ?>
                        </article>
                    </main>
                    <?php wp_footer(); ?>
                </body>
            </html>
            <?php
            exit;
        }, 999);

        //+++

        if (!class_exists('WooCommerce') || !current_user_can('manage_woocommerce') || !is_admin()) {
            return;
        }

        add_filter('plugin_action_links_' . BOTOSCOPE_PLUGIN_NAME, function ($links) {
            $buttons = array(
                '<a href="' . admin_url("edit.php?post_type=product&page={$this->slug}") . '">Botoscope</a>',
                '<a target="_blank" href="https://botoscope.com/documentation"><span class="icon-book"></span>&nbsp;' . esc_html__('Documentation', 'botoscope') . '</a>'
            );

            return array_merge($buttons, $links);
        }, 50);

        add_action('admin_menu', function () {
            if (!current_user_can('manage_woocommerce')) {
                return;
            }

            add_submenu_page('edit.php?post_type=product', 'Botoscope', 'Botoscope', 'manage_woocommerce', $this->slug, function () {
                $args = [];

                $args['woocommerce_product_attributes'] = $this->get_woocommerce_product_attributes();
                $args['botoscope'] = $this;
                $args['languages'] = $this->languages;

                BOTOSCOPE_HELPER::render_html_e(BOTOSCOPE_PATH . 'views/options.php', $args);
            });
        }, 99);

        if (intval($this->controls->get_option('show_bs_button_on_top_menu'))) {
            add_action('admin_bar_menu', function ($admin_bar) {
                if (!current_user_can('manage_woocommerce')) {
                    return;
                }

                $admin_bar->add_menu([
                    'id' => 'botoscope_top_link',
                    'title' => 'Botoscope',
                    'href' => admin_url('edit.php?post_type=product&page=botoscope'),
                    'meta' => [
                        'title' => __('Botoscope: Products Dashboard', 'botoscope'),
                    ],
                ]);
            }, 100);
        }

        add_filter('attachment_fields_to_edit', function ($form_fields, $post) {
            if (strpos($post->post_mime_type, 'image/') === 0) {
                $custom_field = ['botoscope_video_link' =>
                    [
                        'label' => esc_html__('Botoscope telegram video link', 'botoscope'),
                        'input' => 'text',
                        'value' => get_post_meta($post->ID, 'botoscope_video_link', true),
                        'helps' => esc_html__('You can use this field to set a thumbnail preview for your videos in the admin area. This is especially useful for links to videos shared in your dedicated Telegram channel when the videos exceed 25 seconds in length and/or are too large for the Telegram API.', 'botoscope')
                    ]
                ];

                $form_fields = $custom_field + $form_fields; //to make dit first field of the custom ones
            }
            return $form_fields;
        }, 10, 2);

        add_filter('attachment_fields_to_save', function ($post, $attachment) {
            if (isset($attachment['botoscope_video_link'])) {
                update_post_meta($post['ID'], 'botoscope_video_link', sanitize_text_field($attachment['botoscope_video_link']));
            }
            return $post;
        }, 10, 2);
    }

    private function set_ajax_actions() {
        add_action('wp_ajax_botoscope_reset_cache', function () {
            if ($this->is_ajax_request_valid()) {
                $this->reset_cache(sanitize_textarea_field($_REQUEST['cache_name']));
                die('done');
            }

            die('0');
        }, 1);

        add_action('wp_ajax_botoscope_check_nonce', function () {
            if ($this->is_ajax_request_valid()) {
                die('1');
            }

            die('0');
        }, 1);

        add_action('wp_ajax_botoscope_add_row', array($this, 'botoscope_add_row'), 1);
        add_action('wp_ajax_botoscope_edit_cell', array($this, 'botoscope_edit_cell'), 1);
        add_action('wp_ajax_botoscope_get_parent_cell_data', array($this, 'botoscope_get_parent_cell_data'), 1);
        add_action('wp_ajax_botoscope_delete_row', array($this, 'botoscope_delete_row'), 1);
        add_action('wp_ajax_botoscope_search_orders', array($this, 'botoscope_search_orders'), 1);
        add_action('wp_ajax_botoscope_get_translations', array($this, 'botoscope_get_translations'), 1);
        add_action('wp_ajax_botoscope_get_page_data', function () {
            if ($this->is_ajax_request_valid()) {
                $page_num = intval($_REQUEST['page_num']);
                $page_num = $page_num < 0 ? 0 : $page_num;
                die(json_encode($this->get_data(sanitize_text_field($_REQUEST['what']), $page_num, sanitize_text_field($_REQUEST['order_by']), sanitize_text_field($_REQUEST['order']), BOTOSCOPE_HELPER::sanitize_array($_REQUEST['search'] ?? ''))));
            }

            die('0');
        }, 1);
        add_action('wp_ajax_botoscope_translate_string', array($this, 'botoscope_translate_string'), 1);
        add_action('wp_ajax_botoscope_get_sidebar_html', function () {
            if ($this->is_ajax_request_valid()) {
                Hooks::apply_action('botoscope_get_sidebar_html', [], [
                    sanitize_text_field($_REQUEST['what']),
                    sanitize_text_field($_REQUEST['template_name']),
                    intval($_REQUEST['id']),
                ]);
                exit;
            }

            die('0');
        }, 1);

        add_action('wp_ajax_botoscope_edit_row', function () {
            if ($this->is_ajax_request_valid()) {
                $data = isset($_REQUEST['data']) ? $_REQUEST['data'] : [];

                Hooks::apply_action('botoscope_edit_row', [], [
                    sanitize_text_field($_REQUEST['what']),
                    sanitize_text_field($_REQUEST['id']),
                    $data
                ]);
                exit;
            }

            die('0');
        }, 1);

        add_action('wp_ajax_botoscope_set_table_col_positions', function () {
            if ($this->is_ajax_request_valid()) {
                $what = sanitize_text_field($_REQUEST['what']);
                $keys = sanitize_text_field($_REQUEST['keys']);
                update_option("botoscope_{$what}_table_col_positions", $keys);
                exit;
            }

            die('0');
        }, 1);

        add_action('wp_ajax_botoscope_get_table_col_positions', function () {
            if ($this->is_ajax_request_valid()) {
                $what = sanitize_text_field($_REQUEST['what']);
                echo get_option("botoscope_{$what}_table_col_positions", '');
                exit;
            }

            die('0');
        }, 1);
    }

    public function admin_enqueue_scripts() {
        if (isset($_GET['page']) AND $_GET['page'] === $this->slug) {

            add_filter('admin_body_class', function ($classes) {
                $classes .= ' folded'; //todo, place to options

                if (botoscope_is_no_cart()) {
                    $classes .= ' botoscope_is_no_cart';
                }

                return $classes;
            });

            wp_enqueue_media();
            wp_enqueue_script('media-upload');
            wp_enqueue_style('thickbox');
            wp_enqueue_script('thickbox');

            wp_enqueue_script('editor');
            wp_enqueue_script('quicktags');
            wp_enqueue_script('wp-tinymce');
            wp_enqueue_style('editor-style');

            add_filter('script_loader_tag', function ($tag, $handle, $src) {
                if ($handle === $this->slug) {
                    $tag = '<script type="module" src="' . esc_url($src) . '"></script>';
                }
                return $tag;
            }, 10, 3);

            wp_enqueue_style('dashicons');
            wp_enqueue_script('jquery-ui-sortable');
            wp_enqueue_style(
                    'woocommerce-jquery-ui',
                    plugins_url('assets/css/jquery-ui/jquery-ui.min.css', WC_PLUGIN_FILE),
                    [],
                    BOTOSCOPE_VERSION
            );

            wp_enqueue_style('select2-css', WC()->plugin_url() . '/assets/css/select2.css', [], BOTOSCOPE_VERSION);
            wp_enqueue_script('select2-js', WC()->plugin_url() . '/assets/js/select2/select2.full.min.js', ['jquery'], BOTOSCOPE_VERSION, true);

            wp_enqueue_script('botoscope_general', BOTOSCOPE_ASSETS_LINK . 'js/' . 'general.js', [], BOTOSCOPE_VERSION);
            wp_enqueue_script($this->slug, BOTOSCOPE_ASSETS_LINK . 'js/' . $this->slug . '.js', [], BOTOSCOPE_VERSION);
            wp_enqueue_script("{$this->slug}_tabs", BOTOSCOPE_ASSETS_LINK . 'js/tabs.js', [], BOTOSCOPE_VERSION);
            wp_enqueue_script("{$this->slug}_perfect-scrollbar", BOTOSCOPE_ASSETS_LINK . 'js/vendor/perfect-scrollbar.js', [], BOTOSCOPE_VERSION);

            $css = [
                'css/vendor/perfect-scrollbar.css',
                'css/lib/popup-23.css',
                'css/lib/calendar23.css',
                'css/lib/selectm23.css',
                'css/lib/sidebar.css',
                'css/vendor/fontello.css',
                'css/vendor/bootstrap.min.css',
                'css/vendor/growls.css',
                'css/botoscope.css',
                'css/botoscope-mobile.css',
            ];

            foreach ($css as $num => $path) {
                wp_enqueue_style("{$this->slug}_asset_{$num}", BOTOSCOPE_ASSETS_LINK . $path, [], BOTOSCOPE_VERSION);
            }
            ?>
            <script>
                var botoscope_no_bot = <?php echo $this->no_bot ? 1 : 0 ?>;
                var botoscope_url = "<?php echo BOTOSCOPE_LINK ?>";
                var botoscope_custom_ext_url = "<?php echo BOTOSCOPE_CUSTOM_EXT_LINK ?>";
                var botoscope_bot_name = "<?php echo defined('BOTOSCOPE_BOT_NAME') ? BOTOSCOPE_BOT_NAME : '' ?>";
                var botoscope_locale = "<?php echo get_locale() ?>";
                var botoscope_default_language = '<?php echo $this->get_default_language() ?>';
                var botoscope_lang = {};
                var botoscope_is_no_cart =<?php echo botoscope_is_no_cart() ?>;

                var botoscope_is_mobile = <?php echo (wp_is_mobile() ? 1 : 0) ?>;
                var botoscope_default_image = "<?php echo BOTOSCOPE_ASSETS_LINK ?>img/no-image.webp";
                var botoscope_delete_product_without_ask = <?php echo intval($this->controls->get_option('delete_product_without_ask') ?? 0) ?>;

                botoscope_lang.botoscope = "<?php esc_html_e('Botoscope — Your Partner for Telegram Sales Success', 'botoscope') ?>";
                botoscope_lang.after_bear_edit = "<?php esc_html_e('After performing bulk operations, go to the Botoscope system controls and click the “Reset telegram product data cache” button to synchronize with the bot.', 'botoscope') ?>";
                botoscope_lang.type = "<?php esc_html_e('Type', 'botoscope') ?>";
                botoscope_lang.topic = "<?php esc_html_e('Topic', 'botoscope') ?>";
                botoscope_lang.interaction = "<?php esc_html_e('Interaction', 'botoscope') ?>";
                botoscope_lang.unanswered = "<?php esc_html_e('Unanswered', 'botoscope') ?>";
                botoscope_lang.updated = "<?php esc_html_e('Updated', 'botoscope') ?>";
                botoscope_lang.messages = "<?php esc_html_e('Messages', 'botoscope') ?>";
                botoscope_lang.search_orders = "<?php esc_html_e('Search orders', 'botoscope') ?>";
                botoscope_lang.close = "<?php esc_html_e('Close', 'botoscope') ?>";
                botoscope_lang.enter_order_num = "<?php esc_html_e('Please enter order number ...', 'botoscope') ?>";
                botoscope_lang.status = "<?php esc_html_e('Status', 'botoscope') ?>";
                botoscope_lang.total = "<?php esc_html_e('Total', 'botoscope') ?>";
                botoscope_lang.date = "<?php esc_html_e('Date', 'botoscope') ?>";
                botoscope_lang.ticket = "<?php esc_html_e('Ticket', 'botoscope') ?>";
                botoscope_lang.messages_for = "<?php esc_html_e('Messages for', 'botoscope') ?>";
                botoscope_lang.messages_for_order = "<?php esc_html_e('Messages for order', 'botoscope') ?>";
                botoscope_lang.role = "<?php esc_html_e('Role', 'botoscope') ?>";
                botoscope_lang.content = "<?php esc_html_e('Content', 'botoscope') ?>";
                botoscope_lang.time = "<?php esc_html_e('Time', 'botoscope') ?>";
                botoscope_lang.are_you_sure = "<?php esc_html_e('Are you sure?', 'botoscope') ?>";
                botoscope_lang.manage = "<?php esc_html_e('Manage', 'botoscope') ?>";
                botoscope_lang.manage_products = "<?php esc_html_e('Manage products', 'botoscope') ?>";
                botoscope_lang.default = "<?php esc_html_e('Default', 'botoscope') ?>";
                botoscope_lang.title = "<?php esc_html_e('Title', 'botoscope') ?>";
                botoscope_lang.description = "<?php esc_html_e('Description', 'botoscope') ?>";
                botoscope_lang.table = "<?php esc_html_e('Table', 'botoscope') ?>";
                botoscope_lang.active = "<?php esc_html_e('Active', 'botoscope') ?>";
                botoscope_lang.delete = "<?php esc_html_e('Delete', 'botoscope') ?>";
                botoscope_lang.cancel = "<?php esc_html_e('Cancel', 'botoscope') ?>";
                botoscope_lang.charts_for = "<?php esc_html_e('Charts for', 'botoscope') ?>";
                botoscope_lang.height = "<?php esc_html_e('Height', 'botoscope') ?>";
                botoscope_lang.neck = "<?php esc_html_e('Neck', 'botoscope') ?>";
                botoscope_lang.shoulder = "<?php esc_html_e('Shoulder', 'botoscope') ?>";
                botoscope_lang.breast = "<?php esc_html_e('Breast', 'botoscope') ?>";
                botoscope_lang.waist = "<?php esc_html_e('Waist', 'botoscope') ?>";
                botoscope_lang.hip = "<?php esc_html_e('Hip', 'botoscope') ?>";
                botoscope_lang.arm = "<?php esc_html_e('Arm', 'botoscope') ?>";
                botoscope_lang.leg_length_from_waist = "<?php esc_html_e('Leg length from waist', 'botoscope') ?>";
                botoscope_lang.append_new_row = "<?php esc_html_e('append new row', 'botoscope') ?>";
                botoscope_lang.order = "<?php esc_html_e('Order', 'botoscope') ?>";
                botoscope_lang.price = "<?php esc_html_e('Price', 'botoscope') ?>";
                botoscope_lang.minimum = "<?php esc_html_e('Minimum', 'botoscope') ?>";
                botoscope_lang.product = "<?php esc_html_e('Product', 'botoscope') ?>";
                botoscope_lang.product_downloads = "<?php esc_html_e('Product downloads', 'botoscope') ?>";
                botoscope_lang.terms = "<?php esc_html_e('Terms', 'botoscope') ?>";
                botoscope_lang.search_by_title_or_sku = "<?php esc_html_e('search by title or sku or ID ...', 'botoscope') ?>";
                botoscope_lang.add = "<?php esc_html_e('Add', 'botoscope') ?>";
                botoscope_lang.select_media = "<?php esc_html_e('Select media for the product', 'botoscope') ?>";
                botoscope_lang.media = "<?php esc_html_e('Media', 'botoscope') ?>";
                botoscope_lang.sale = "<?php esc_html_e('Sale', 'botoscope') ?>";
                botoscope_lang.sku = "<?php esc_html_e('SKU', 'botoscope') ?>";
                botoscope_lang.category = "<?php esc_html_e('Category', 'botoscope') ?>";
                botoscope_lang.select_category = "<?php esc_html_e('select product category ...', 'botoscope') ?>";
                botoscope_lang.select_terms = "<?php esc_html_e('select terms', 'botoscope') ?>";
                botoscope_lang.select_attribute = "<?php esc_html_e('Select attribute', 'botoscope') ?>";
                botoscope_lang.attribute = "<?php esc_html_e('Attribute', 'botoscope') ?>";
                botoscope_lang.display = "<?php esc_html_e('Display', 'botoscope') ?>";
                botoscope_lang.inline = "<?php esc_html_e('Inline', 'botoscope') ?>";
                botoscope_lang.formula = "<?php esc_html_e('Formula', 'botoscope') ?>";
                botoscope_lang.icon = "<?php esc_html_e('Icon', 'botoscope') ?>";
                botoscope_lang.name = "<?php esc_html_e('Name', 'botoscope') ?>";
                botoscope_lang.address = "<?php esc_html_e('Address', 'botoscope') ?>";
                botoscope_lang.details = "<?php esc_html_e('Details', 'botoscope') ?>";
                botoscope_lang.shipping = "<?php esc_html_e('Shipping', 'botoscope') ?>";
                botoscope_lang.formulas = "<?php esc_html_e('Formulas', 'botoscope') ?>";
                botoscope_lang.formulas_for = "<?php esc_html_e('Formulas for', 'botoscope') ?>";
                botoscope_lang.formula = "<?php esc_html_e('Formula', 'botoscope') ?>";
                botoscope_lang.mar_ext_should_active = "<?php esc_html_e('Marketing strategies ext should be activated!', 'botoscope') ?>";
                botoscope_lang.products = "<?php esc_html_e('Products', 'botoscope') ?>";
                botoscope_lang.excluded_products = "<?php esc_html_e('Excluded Products', 'botoscope') ?>";
                botoscope_lang.products_for = "<?php esc_html_e('Products for', 'botoscope') ?>";
                botoscope_lang.groups_for = "<?php esc_html_e('Groups for', 'botoscope') ?>";
                botoscope_lang.select_strategia = "<?php esc_html_e('Select strategia', 'botoscope') ?>";
                botoscope_lang.select_date = "<?php esc_html_e('select date', 'botoscope') ?>";
                botoscope_lang.strategia_id = "<?php esc_html_e('Strategia', 'botoscope') ?>";
                botoscope_lang.time_start = "<?php esc_html_e('Start', 'botoscope') ?>";
                botoscope_lang.time_finish = "<?php esc_html_e('Finish', 'botoscope') ?>";
                botoscope_lang.key = "<?php esc_html_e('Key', 'botoscope') ?>";
                botoscope_lang.original = "<?php esc_html_e('Original', 'botoscope') ?>";
                botoscope_lang.customized = "<?php esc_html_e('Customized', 'botoscope') ?>";
                botoscope_lang.code = "<?php esc_html_e('Code', 'botoscope') ?>";
                botoscope_lang.amount = "<?php esc_html_e('Amount', 'botoscope') ?>";
                botoscope_lang.usage_limit = "<?php esc_html_e('Limit', 'botoscope') ?>";
                botoscope_lang.maximum = "<?php esc_html_e('Maximum', 'botoscope') ?>";
                botoscope_lang.expiry = "<?php esc_html_e('Expiry', 'botoscope') ?>";
                botoscope_lang.enter3_to_start_search = "<?php esc_html_e('Please enter 3 or more characters to start searching, or type an product ID (e.g. 123 or v123(for variations of variable product))...', 'botoscope') ?>";
                botoscope_lang.child_terms = "<?php esc_html_e('Child terms', 'botoscope') ?>";
                botoscope_lang.translated = "<?php esc_html_e('Translated', 'botoscope') ?>";
                botoscope_lang.settings = "<?php esc_html_e('Settings', 'botoscope') ?>";
                botoscope_lang.settings_for = "<?php esc_html_e('Settings for', 'botoscope') ?>";
                botoscope_lang.value = "<?php esc_html_e('Value', 'botoscope') ?>";
                botoscope_lang.video = "<?php esc_html_e('Video', 'botoscope') ?>";
                botoscope_lang.bot_languages = "<?php esc_html_e('Bot Languages', 'botoscope') ?>";
                botoscope_lang.reset_products_cache️ = "<?php esc_html_e('Syncing products with Telegram', 'botoscope') ?>";
                botoscope_lang.reset_system_cache️ = "<?php esc_html_e('Full data synchronization with Telegram', 'botoscope') ?>";
                botoscope_lang.reset_options_cache️ = "<?php esc_html_e('Syncing site settings with Telegram', 'botoscope') ?>";
                botoscope_lang.percent_to_cart = "<?php esc_html_e('percent to cart', 'botoscope') ?>";
                botoscope_lang.fixed_to_cart = "<?php esc_html_e('fixed to cart', 'botoscope') ?>";
                botoscope_lang.percent_to_selected_product = "<?php esc_html_e('percent to selected product', 'botoscope') ?>";
                botoscope_lang.fixed_to_selected_product = "<?php esc_html_e('fixed to selected product', 'botoscope') ?>";
                botoscope_lang.send_message = "<?php esc_html_e('send message', 'botoscope') ?>";
                botoscope_lang.in_stock = "<?php esc_html_e('in stock', 'botoscope') ?>";
                botoscope_lang.out_of_stock = "<?php esc_html_e('out of stock', 'botoscope') ?>";
                botoscope_lang.delivery_methods = "<?php esc_html_e('delivery methods', 'botoscope') ?>";
                botoscope_lang.wrong_sale_price = "<?php esc_html_e('Sale price should be lower than the regular price!', 'botoscope') ?>";
                botoscope_lang.provide_some_text = "<?php esc_html_e('You should provide some text', 'botoscope') ?>";
                botoscope_lang.saved = "<?php esc_html_e('Saved', 'botoscope') ?>";
                botoscope_lang.saving = "<?php esc_html_e('Saving ...', 'botoscope') ?>";
                botoscope_lang.file_url = "<?php esc_html_e('File URL', 'botoscope') ?>";
                botoscope_lang.no_data = "<?php esc_html_e('No data', 'botoscope') ?>";
                botoscope_lang.loading = "<?php esc_html_e('Loading ...', 'botoscope') ?>";
                botoscope_lang.done = "<?php esc_html_e('Done', 'botoscope') ?>";
                botoscope_lang.variation = "<?php esc_html_e('Variation', 'botoscope') ?>";
                botoscope_lang.variations = "<?php esc_html_e('Variations', 'botoscope') ?>";
                botoscope_lang.variations_of = "<?php esc_html_e('Variations of product', 'botoscope') ?>";
                botoscope_lang.variations_for = "<?php esc_html_e('Variations for product', 'botoscope') ?>";
                botoscope_lang.variation_of = "<?php esc_html_e('Variation of product', 'botoscope') ?>";
                botoscope_lang.select_possible_variation = "<?php esc_html_e('Select possible variation', 'botoscope') ?>";
                botoscope_lang.no_free_possible_variations = "<?php esc_html_e('There are no free possible combinations', 'botoscope') ?>";
                botoscope_lang.select = "<?php esc_html_e('Select', 'botoscope') ?>";
                botoscope_lang.combination = "<?php esc_html_e('Combination', 'botoscope') ?>";
                botoscope_lang.wrong_combination = "<?php esc_html_e('Combination of attributes should be unique', 'botoscope') ?>";
                botoscope_lang.attributes = "<?php esc_html_e('Attributes', 'botoscope') ?>";
                botoscope_lang.not_selected = "<?php esc_html_e('not selected', 'botoscope') ?>";
                botoscope_lang.new_attribute_taxonomy = "<?php esc_html_e('New attribute', 'botoscope') ?>";
                botoscope_lang.new_taxonomy = "<?php esc_html_e('New taxonomy', 'botoscope') ?>";
                botoscope_lang.translating = "<?php esc_html_e('Translating ...', 'botoscope') ?>";
                botoscope_lang.require_openai_key = "<?php esc_html_e('Place into settings openai API key!', 'botoscope') ?>";
                botoscope_lang.append_formula = "<?php esc_html_e('append new formula', 'botoscope') ?>";
                botoscope_lang.select_attributes_and_terms = "<?php esc_html_e('Kindly select the product attributes and their corresponding terms first, and then click save button', 'botoscope') ?>";
                botoscope_lang.select_prod_type = "<?php esc_html_e('Select product type', 'botoscope') ?>";
                botoscope_lang.select_prod_category = "<?php esc_html_e('Select product category', 'botoscope') ?>";
                botoscope_lang.simple = "<?php esc_html_e('Simple', 'botoscope') ?>";
                botoscope_lang.simple_virtual = "<?php esc_html_e('Simple virtual', 'botoscope') ?>";
                botoscope_lang.simple_virtual_downloadable = "<?php esc_html_e('Simple virtual downloadable', 'botoscope') ?>";
                botoscope_lang.simple_media_casting = "<?php esc_html_e('Simple media casting', 'botoscope') ?>";
                botoscope_lang.external = "<?php esc_html_e('External', 'botoscope') ?>";
                botoscope_lang.grouped = "<?php esc_html_e('Grouped', 'botoscope') ?>";
                botoscope_lang.variable = "<?php esc_html_e('Variable', 'botoscope') ?>";
                botoscope_lang.button = "<?php esc_html_e('Button', 'botoscope') ?>";
                botoscope_lang.switcher = "<?php esc_html_e('Switcher', 'botoscope') ?>";
                botoscope_lang.max_files_count = "<?php esc_html_e('The maximum allowed number of files is %s. Only %s of the selected files have been added.', 'botoscope') ?>";
                botoscope_lang.max_files_count_no_added = "<?php esc_html_e('You’ve reached the maximum file limit of %s. No more files can be added.', 'botoscope') ?>";
                botoscope_lang.orders = "<?php esc_html_e('Orders', 'botoscope') ?>";
                botoscope_lang.reports = "<?php esc_html_e('Reports', 'botoscope') ?>";
                botoscope_lang.gallery = "<?php esc_html_e('Gallery', 'botoscope') ?>";
                botoscope_lang.bulk_editing = "<?php esc_html_e('Bulk Editing', 'botoscope') ?>";
                botoscope_lang.currency = "<?php esc_html_e('Currency', 'botoscope') ?>";
                botoscope_lang.filter = "<?php esc_html_e('Filter', 'botoscope') ?>";
                botoscope_lang.log_out = "<?php esc_html_e('Log out', 'botoscope') ?>";
                botoscope_lang.enter_products_count_to_create = "<?php esc_html_e('Enter the number of products you want to create', 'botoscope') ?>";
                botoscope_lang.append_to_group = "<?php esc_html_e('to group', 'botoscope') ?>";
                botoscope_lang.enter_group_product_id = "<?php esc_html_e('Enter the ID of the group product. Enter 0 to detach', 'botoscope') ?>";
                botoscope_lang.select_audio = "<?php esc_html_e('Select audio file', 'botoscope') ?>";
                botoscope_lang.add_audio = "<?php esc_html_e('Add audio', 'botoscope') ?>";
                botoscope_lang.select_image = "<?php esc_html_e('Select image', 'botoscope') ?>";
                botoscope_lang.add_image = "<?php esc_html_e('Add image', 'botoscope') ?>";
                botoscope_lang.yes = "<?php esc_html_e('Yes', 'botoscope') ?>";
                botoscope_lang.no = "<?php esc_html_e('No', 'botoscope') ?>";
                botoscope_lang.insert_html = "<?php esc_html_e('Insert HTML', 'botoscope') ?>";
                botoscope_lang.enter_html_code = "<?php esc_html_e('Enter HTML code', 'botoscope') ?>";
                botoscope_lang.add_all = "<?php esc_html_e('Add all', 'botoscope') ?>";
                botoscope_lang.composition = "<?php esc_html_e('Composition', 'botoscope') ?>";
                botoscope_lang.create = "<?php esc_html_e('Create', 'botoscope') ?>";
                botoscope_lang.clone = "<?php esc_html_e('Clone', 'botoscope') ?>";
                botoscope_lang.create_meta_field = "<?php esc_html_e('Create meta field', 'botoscope') ?>";
                botoscope_lang.append_meta_field = "<?php esc_html_e('Append meta field', 'botoscope') ?>";
                botoscope_lang.product_meta = "<?php esc_html_e('Product meta', 'botoscope') ?>";
                botoscope_lang.products_meta_gallery = "<?php esc_html_e('Products meta gallery', 'botoscope') ?>";
                botoscope_lang.unit_of_measurement = "<?php esc_html_e('Measure', 'botoscope') ?>";
                botoscope_lang.append = "<?php esc_html_e('Append', 'botoscope') ?>";
                botoscope_lang.create_meta_pack = "<?php esc_html_e('Create meta pack', 'botoscope') ?>";
                botoscope_lang.load_meta_pack = "<?php esc_html_e('Load meta pack', 'botoscope') ?>";
                botoscope_lang.meta_packs = "<?php esc_html_e('Meta packs', 'botoscope') ?>";
                botoscope_lang.apply = "<?php esc_html_e('Apply', 'botoscope') ?>";
                botoscope_lang.meta_position_media = "<?php esc_html_e('Show on media gallery', 'botoscope') ?>";
                botoscope_lang.meta_position_description = "<?php esc_html_e('Show under description', 'botoscope') ?>";
                botoscope_lang.added = "<?php esc_html_e('Added', 'botoscope') ?>";
                botoscope_lang.analytics = "<?php esc_html_e('Analytics', 'botoscope') ?>";
                botoscope_lang.pages = "<?php esc_html_e('Pages', 'botoscope') ?>";
                botoscope_lang.refund_policy = "<?php esc_html_e('Refund policy', 'botoscope') ?>";
                botoscope_lang.privacy_policy = "<?php esc_html_e('Privacy policy', 'botoscope') ?>";
                botoscope_lang.shipping_policy = "<?php esc_html_e('Shipping policy', 'botoscope') ?>";
                botoscope_lang.edit = "<?php esc_html_e('Edit', 'botoscope') ?>";
                botoscope_lang.meta_delete = "<?php esc_html_e('Are you sure? All metadata attached to products will also be deleted. This action cannot be undone!', 'botoscope') ?>";
                botoscope_lang.message = "<?php esc_html_e('Message', 'botoscope') ?>";
                botoscope_lang.sent = "<?php esc_html_e('Sent', 'botoscope') ?>";
                botoscope_lang.tools = "<?php esc_html_e('Tools', 'botoscope') ?>";
                botoscope_lang.products_import = "<?php esc_html_e('Products import', 'botoscope') ?>";
                botoscope_lang.products_export = "<?php esc_html_e('Products export', 'botoscope') ?>";
                botoscope_lang.booking_slots_for = "<?php esc_html_e('Booking slots for product', 'botoscope') ?>";
                botoscope_lang.start_h = "<?php esc_html_e('Start hour', 'botoscope') ?>";
                botoscope_lang.start_m = "<?php esc_html_e('Start minute', 'botoscope') ?>";
                botoscope_lang.finish_h = "<?php esc_html_e('Finish hour', 'botoscope') ?>";
                botoscope_lang.finish_m = "<?php esc_html_e('Finish minute', 'botoscope') ?>";
                botoscope_lang.mon = "<?php esc_html_e('Mon', 'botoscope') ?>";
                botoscope_lang.tue = "<?php esc_html_e('Tue', 'botoscope') ?>";
                botoscope_lang.wed = "<?php esc_html_e('Wed', 'botoscope') ?>";
                botoscope_lang.thu = "<?php esc_html_e('Thu', 'botoscope') ?>";
                botoscope_lang.fri = "<?php esc_html_e('Fri', 'botoscope') ?>";
                botoscope_lang.sat = "<?php esc_html_e('Sat', 'botoscope') ?>";
                botoscope_lang.sun = "<?php esc_html_e('Sun', 'botoscope') ?>";
                botoscope_lang.hold = "<?php esc_html_e('Hold', 'botoscope') ?>";
                botoscope_lang.hold_customer = "<?php esc_html_e('Hold customer to slot', 'botoscope') ?>";
                botoscope_lang.booking_is_on = "<?php esc_html_e('Booking system is on', 'botoscope') ?>";
                botoscope_lang.booking_is_off = "<?php esc_html_e('Booking system is off', 'botoscope') ?>";
                botoscope_lang.capacity = "<?php esc_html_e('Capacity', 'botoscope') ?>";
                botoscope_lang.set_start = "<?php esc_html_e('set start', 'botoscope') ?>";
                botoscope_lang.set_finish = "<?php esc_html_e('set finish', 'botoscope') ?>";
                botoscope_lang.test_mode = "<?php esc_html_e('Test mode', 'botoscope') ?>";
                botoscope_lang.deactivated = "<?php esc_html_e('Deactivated', 'botoscope') ?>";
                botoscope_lang.help = "<?php esc_html_e('Help', 'botoscope') ?>";

                botoscope_lang.month_names = [
                    "<?php esc_html_e('January', 'botoscope'); ?>",
                    "<?php esc_html_e('February', 'botoscope'); ?>",
                    "<?php esc_html_e('March', 'botoscope'); ?>",
                    "<?php esc_html_e('April', 'botoscope'); ?>",
                    "<?php esc_html_e('May', 'botoscope'); ?>",
                    "<?php esc_html_e('June', 'botoscope'); ?>",
                    "<?php esc_html_e('July', 'botoscope'); ?>",
                    "<?php esc_html_e('August', 'botoscope'); ?>",
                    "<?php esc_html_e('September', 'botoscope'); ?>",
                    "<?php esc_html_e('October', 'botoscope'); ?>",
                    "<?php esc_html_e('November', 'botoscope'); ?>",
                    "<?php esc_html_e('December', 'botoscope'); ?>"
                ];

                botoscope_lang.month_names_short = [
                    "<?php esc_html_e('Jan', 'botoscope'); ?>",
                    "<?php esc_html_e('Feb', 'botoscope'); ?>",
                    "<?php esc_html_e('Mar', 'botoscope'); ?>",
                    "<?php esc_html_e('Apr', 'botoscope'); ?>",
                    "<?php esc_html_e('May', 'botoscope'); ?>",
                    "<?php esc_html_e('Jun', 'botoscope'); ?>",
                    "<?php esc_html_e('Jul', 'botoscope'); ?>",
                    "<?php esc_html_e('Aug', 'botoscope'); ?>",
                    "<?php esc_html_e('Sep', 'botoscope'); ?>",
                    "<?php esc_html_e('Oct', 'botoscope'); ?>",
                    "<?php esc_html_e('Nov', 'botoscope'); ?>",
                    "<?php esc_html_e('Dec', 'botoscope'); ?>"
                ];

                botoscope_lang.day_names = [
                    "<?php esc_html_e('Mo', 'botoscope'); ?>",
                    "<?php esc_html_e('Tu', 'botoscope'); ?>",
                    "<?php esc_html_e('We', 'botoscope'); ?>",
                    "<?php esc_html_e('Th', 'botoscope'); ?>",
                    "<?php esc_html_e('Fr', 'botoscope'); ?>",
                    "<?php esc_html_e('Sa', 'botoscope'); ?>",
                    "<?php esc_html_e('Su', 'botoscope'); ?>"
                ];

                var botoscope_site_url = "<?php echo get_site_url() ?>/";
            </script>
            <?php
        }
    }

    //+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

    public function get_default_language() {
        return $this->controls->get_default_language() ?? substr(get_locale(), 0, 2);
    }

    private function get_data($what, $page_num = 0, $order_by = '', $order = '', $search = []) {
        $res = [];

        if ($order_by === 'oid') {
            $order_by = 'id';
        }

        if (property_exists($this, $what) && $this->$what && method_exists($this->$what, 'get')) {
            $this->$what->search = $search;
            $res = $this->$what->get($page_num, compact('order_by', 'order', 'search'));
            if (property_exists($this->$what, 'found_posts') && $this->$what->found_posts !== -1) {
                $res = ['posts' => $res, 'found_posts' => $this->$what->found_posts];
            }
        } else {
            $res = Hooks::apply_action('botoscope_get_page_data', [], [$what, $page_num, $order_by, $order, $search]);
        }

        return $res;
    }

    //for botoscope bridge plugin
    public function get_active($what, $more_data = []) {
        $data = $this->get_data($what);

        if (property_exists($this, $what) && $this->$what && method_exists($this->$what, 'get_active')) {
            $res = $this->$what->get_active($more_data);
        } else {
            $res = array_values(array_filter($data, function ($item) {
                        return isset($item['is_active']) && intval($item['is_active']) === 1;
                    }));
        }

        return $res;
    }

    //ajax
    public function botoscope_add_row() {
        if ($this->is_ajax_request_valid()) {
            $what = sanitize_text_field($_REQUEST['what']);
            $parent_row_id = isset($_REQUEST['additional_params']['parent_row_id']) ? intval($_REQUEST['additional_params']['parent_row_id']) : 0;
            $content = isset($_REQUEST['additional_params']['content']) ? sanitize_textarea_field($_REQUEST['additional_params']['content']) : '';

            if (property_exists($this, $what) && $this->$what && method_exists($this->$what, 'create')) {
                $res = $this->$what->create();
            } else {
                $res = Hooks::apply_action('botoscope_add_row', [], [$what, $parent_row_id, $content]);
            }

            die(json_encode($res));
        }
    }

    //ajax
    public function botoscope_edit_cell() {
        if ($this->is_ajax_request_valid()) {

            $key = esc_html__($_REQUEST['key'] ?? '');
            $id = isset($_REQUEST['id']) ? is_numeric($_REQUEST['id']) ? intval($_REQUEST['id']) : sanitize_text_field($_REQUEST['id']) : 0;
            $value = $_REQUEST['value'] ?? '';
            $what = esc_html__($_REQUEST['what']);

            if (property_exists($this, $what) && $this->$what && method_exists($this->$what, 'update')) {

                //!!fix
                if ($value === 'null') {
                    $value = null;
                }

                $this->$what->update($id, $key, $value, $_REQUEST);
                if ($this->$what->synhronize_cache) {
                    $this->reset_cache($what);
                }
            }

            Hooks::apply_action('botoscope_edit_cell', [], [$what, $id, $key, $value, $_REQUEST]);

            die('done');
        }
    }

    //ajax
    public function botoscope_get_parent_cell_data() {
        if ($this->is_ajax_request_valid()) {
            $parent_app = sanitize_text_field($_REQUEST['parent_app']);
            $parent_cell_name = sanitize_text_field($_REQUEST['parent_cell_name']);
            $parent_row_id = is_numeric($_REQUEST['parent_row_id']) ? intval($_REQUEST['parent_row_id']) : sanitize_text_field($_REQUEST['parent_row_id']);

            $res = Hooks::apply_action('botoscope_get_parent_cell_data', [], [$parent_app, $parent_row_id, $parent_cell_name]);
            die(json_encode($res));
        }
    }

    //ajax
    public function botoscope_delete_row() {
        if ($this->is_ajax_request_valid()) {

            $what = esc_html__($_REQUEST['what']);
            $row_id = intval($_REQUEST['row_id']);

            if (isset($_REQUEST['additional_params']['parent_row_id'])) {
                $parent_row_id = intval($_REQUEST['additional_params']['parent_row_id'] ?? 0);
            } else {
                $parent_row_id = intval($_REQUEST['parent_row_id'] ?? 0);
            }

            if (property_exists($this, $what) && $this->$what && method_exists($this->$what, 'delete')) {
                $this->$what->delete($row_id);
                if ($this->$what->synhronize_cache) {
                    $this->reset_cache($what);
                }
            }

            Hooks::apply_action('botoscope_delete_row', [], [$what, $row_id, $parent_row_id]);

            die('done');
        }
    }

    //ajax
    public function botoscope_get_translations() {
        if ($this->is_ajax_request_valid()) {
            $res = [];

            $parent_app = sanitize_text_field($_REQUEST['parent_app']);
            $parent_cell_name = sanitize_text_field($_REQUEST['parent_cell_name']);
            $parent_row_id = is_numeric($_REQUEST['parent_row_id']) ? intval($_REQUEST['parent_row_id']) : sanitize_text_field($_REQUEST['parent_row_id']);

            $active_languages = $this->controls->get_active_languages();

            if (!empty($active_languages)) {
                foreach ($active_languages as $language) {
                    $translation = $this->translations->get_translation($language, $parent_app, $parent_row_id, $parent_cell_name);

                    $res[] = [
                        'id' => $translation['id'],
                        'language' => $language,
                        'value' => strval($translation['value'])
                    ];
                }
            }


            die(json_encode($res));
        }
    }

    public function get_woocommerce_product_attributes() {
        $attributes = wc_get_attribute_taxonomies();
        $attributes_array = [];

        foreach ($attributes as $attribute) {
            $attribute = (array) $attribute;
            $taxonomy = 'pa_' . $attribute['attribute_name'];

            //if (taxonomy_exists($taxonomy)) {
            $attributes_array[] = [
                'name' => strip_tags($attribute['attribute_label']),
                'slug' => $taxonomy
            ];
            //}
        }

        return $attributes_array;
    }

    public function do_command($chat_id, $command, $params) {

        if ($this->no_bot) {
            return;
        }

        $data = [];
        $data['chat_id'] = $chat_id;
        $data['command'] = $command;
        $data['data'] = $params;
        $data['encrypted_pass'] = defined('BOTOSCOPE_CLIENT_PASS') ? BOTOSCOPE_HELPER::encrypt_value(BOTOSCOPE_CLIENT_PASS, BOTOSCOPE_CLIENT_API_KEY) : '';

        /*
          $context = stream_context_create(array(
          'http' => array(
          'header' => "Content-Type: application/json\r\n",
          'method' => 'POST',
          'content' => json_encode($data),
          'ignore_errors' => true, // Add for debugging
          ),
          ));
         *
         */

        //return file_get_contents(BOTOSCOPE_PROXY_SERVER, false, $context);
        $client_api_key = defined('BOTOSCOPE_CLIENT_API_KEY') ? BOTOSCOPE_CLIENT_API_KEY : '';
        if (defined('BOTOSCOPE_PROXY_SERVER')) {
            return BOTOSCOPE_HELPER::http_request(BOTOSCOPE_PROXY_SERVER, [
                        'headers' => [
                            'Content-Type' => 'application/json',
                            'Client-Api-Key' => $client_api_key
                        ],
                        'method' => 'POST',
                        'body' => json_encode($data)
            ]);
        }
    }

    public function reset_cache($cache_name) {
        $this->do_command(-1, 'reset_cache', [
            'cache_name' => $cache_name
        ]);
    }

    public function botoscope_search_orders() {
        if ($this->is_ajax_request_valid()) {

            $res = [];

            $search_term = sanitize_text_field($_REQUEST['value']);
            if (!ctype_digit($search_term)) {
                die('0');
            }

            global $wpdb;
            $table_name = $wpdb->prefix . 'wc_orders';

            $order_id = $wpdb->get_var(
                    $wpdb->prepare("
            SELECT id
            FROM $table_name
            WHERE id = %d
        ", $search_term)
            );

            if ($order_id) {
                $order = wc_get_order($order_id);

                if ($order) {
                    $res[] = [
                        'id' => $order->get_id(),
                        'status' => wc_get_order_status_name($order->get_status()),
                        'total' => $order->get_total(),
                        'date' => $order->get_date_created()->date('Y-m-d H:i:s')
                    ];
                }
            }

            die(json_encode($res));
        }
    }

    //ajax
    public function botoscope_translate_string() {

        $api_key = $this->controls->get_option('openai_api_key');

        $text = sanitize_textarea_field($_REQUEST['value']);
        $target_language = trim(sanitize_text_field($_REQUEST['selected_language']));
        $url = "https://api.openai.com/v1/chat/completions";

        $headers = [
            "Content-Type: application/json",
            "Authorization: Bearer $api_key"
        ];

        $data = [
            "model" => "gpt-4-turbo",
            "messages" => [
                [
                    "role" => "system",
                    "content" => "You are a professional translator. Translate the provided text into {$target_language}. Return only the translated text with exactly the same formatting as the original, including line breaks, empty lines, and paragraph spacing. Ensure that the first letter of each sentence is properly capitalized. Keep the original syntax and context of emoji symbols, if present."
                ],
                [
                    "role" => "user",
                    "content" => $text
                ]
            ],
            "temperature" => 0.5
        ];

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));

        $response = curl_exec($ch);
        curl_close($ch);

        $response_data = json_decode($response, true);
        wp_die($response_data["choices"][0]["message"]["content"] ?? "-1");
    }

    public function is_ajax_request_valid() {
        return !(!current_user_can('manage_woocommerce') || !isset($_REQUEST['botoscope_form_nonce']) || !wp_verify_nonce($_REQUEST['botoscope_form_nonce'], 'botoscope_form_nonce'));
    }
}

//!!
add_filter('woocommerce_currencies', function ($currencies) {
    $currencies['XTR'] = __('Telegram Stars', 'woocommerce');
    return $currencies;
});

add_filter('woocommerce_currency_symbol', function ($symbol, $currency) {
    if ($currency === 'XTR') {
        return '⭐';
    }
    return $symbol;
}, 10, 2);

add_action('woocommerce_init', function () {
    global $WOOCS;
    if ($WOOCS) {
        //fix to avoid currency change in botoscope admin panel!!
        $WOOCS->set_currency($WOOCS->default_currency);
    }

    botoscope_init_product_types();

    $GLOBALS['Botoscope'] = $Botoscope = new Botoscope();
    add_action('init', array($Botoscope, 'init'), 99999);
    include_once 'wp_hooks.php';
});

//lets register system taxonomies
add_action('init', function () {
    check_botoscope_pages();
    $existing_taxonomies = get_option('botoscope_taxonomies', []);
    if (!empty($existing_taxonomies)) {
        foreach ($existing_taxonomies as $data) {
            register_taxonomy($data['slug'], 'product', [
                'label' => $data['name'],
                'hierarchical' => true,
                'public' => true,
                'show_ui' => true,
                'show_in_rest' => true,
                'rewrite' => ['slug' => $data['slug']],
            ]);
        }
    }
});

//+++
//important to let next pages always exists
function check_botoscope_pages() {

    $page_slug = 'botoscope-refund-policy';
    $page = get_page_by_path($page_slug, OBJECT, 'page');
    if (!$page) {
        wp_insert_post([
            'post_title' => esc_html__('Botoscope refund policy', 'botoscope'),
            'post_name' => $page_slug,
            'post_content' => esc_html__('Here you can specify the text of your return policy', 'botoscope'),
            'post_status' => 'publish',
            'post_type' => 'page'
        ]);
    }

    //+++

    $page_slug = 'botoscope-privacy-policy';
    $page = get_page_by_path($page_slug, OBJECT, 'page');
    if (!$page) {
        wp_insert_post([
            'post_title' => esc_html__('Botoscope privacy policy', 'botoscope'),
            'post_name' => $page_slug,
            'post_content' => esc_html__('Here you can specify the text of your privacy policy', 'botoscope'),
            'post_status' => 'publish',
            'post_type' => 'page'
        ]);
    }

    //+++

    $page_slug = 'botoscope-shipping-policy';
    $page = get_page_by_path($page_slug, OBJECT, 'page');
    if (!$page) {
        wp_insert_post([
            'post_title' => esc_html__('Botoscope shipping policy', 'botoscope'),
            'post_name' => $page_slug,
            'post_content' => esc_html__('Here you can specify the text of your shipping policy', 'botoscope'),
            'post_status' => 'publish',
            'post_type' => 'page'
        ]);
    }

    //+++

    $page_slug = 'botoscope-media-casting';
    $page = get_page_by_path($page_slug, OBJECT, 'page');
    if (!$page) {
        wp_insert_post([
            'post_title' => esc_html__('Botoscope media casting', 'botoscope'),
            'post_name' => $page_slug,
            'post_content' => '[botoscope_media_casting]',
            'post_status' => 'publish',
            'post_type' => 'page'
        ]);
    }

    //+++

    $page_slug = 'botoscope-product-details';
    $page = get_page_by_path($page_slug, OBJECT, 'page');
    if (!$page) {
        wp_insert_post([
            'post_title' => esc_html__('Botoscope product details', 'botoscope'),
            'post_name' => $page_slug,
            'post_content' => '[botoscope_product_details]',
            'post_status' => 'publish',
            'post_type' => 'page'
        ]);
    }

    //+++

    $page_slug = 'botoscope-filter';
    $page = get_page_by_path($page_slug, OBJECT, 'page');
    if (!$page) {
        wp_insert_post([
            'post_title' => esc_html__('Botoscope filter', 'botoscope'),
            'post_name' => $page_slug,
            'post_content' => '[woof autosubmit=0 redirect="' . get_site_url(null, $page_slug) . '" btn_position=tb]',
            'post_status' => 'publish',
            'post_type' => 'page'
        ]);
    }

    //+++

    $page_slug = 'botoscope-variation-gallery';
    $page = get_page_by_path($page_slug, OBJECT, 'page');
    if (!$page) {
        wp_insert_post([
            'post_title' => esc_html__('Botoscope variation gallery', 'botoscope'),
            'post_name' => $page_slug,
            'post_content' => '[botoscope_variation_gallery]',
            'post_status' => 'publish',
            'post_type' => 'page'
        ]);
    }

    //+++

    $page_slug = 'botoscope-thank-you';
    $page = get_page_by_path($page_slug, OBJECT, 'page');
    if (!$page) {
        wp_insert_post([
            'post_title' => esc_html__('Thank you', 'botoscope'),
            'post_name' => $page_slug,
            'post_content' => '🙏😊🎉💖👍',
            'post_status' => 'publish',
            'post_type' => 'page'
        ]);
    }


    //+++
    /*
      $page_slug = 'products-in-telegram';
      $page = get_page_by_path($page_slug, OBJECT, 'page');
      if (!$page) {
      wp_insert_post([
      'post_title' => esc_html__('Products in Telegram', 'botoscope'),
      'post_name' => $page_slug,
      'post_content' => '[botoscope_telegram_products]',
      'post_status' => 'publish',
      'post_type' => 'page'
      ]);
      }
     *
     */
}

function botoscope_is_no_cart() {
    return intval(get_option('botoscope_disable_cart_checkout', 0)) ? 1 : 0;
}

function botoscope_init_product_types() {
    add_filter('product_type_selector', function ($types) {
        $types['botoscope_simple_virtual'] = __('Botoscope Simple Virtual', 'botoscope');
        $types['botoscope_simple_virtual_downloadable'] = __('Botoscope Simple Virtual Downloadable', 'botoscope');
        $types['botoscope_simple_media_casting'] = __('Botoscope Simple Media Casting', 'botoscope');

        return $types;
    });

    if (!class_exists('WC_Product_Botoscope_Simple_Virtual')) {

        class WC_Product_Botoscope_Simple_Virtual extends WC_Product_Simple {

            protected $product_type = 'botoscope_simple_virtual';
            protected $virtual = 'yes';

            public function __construct($product = 0) {
                parent::__construct($product);
                $this->product_type = 'botoscope_simple_virtual';
                $this->virtual = 'yes';
            }

            public function get_type() {
                return 'botoscope_simple_virtual';
            }

            public function is_virtual() {
                return true;
            }

            public function is_downloadable($context = 'view') {
                return false;
            }
        }

        class WC_Product_Botoscope_Simple_Virtual_Downloadable extends WC_Product_Simple {

            protected $product_type = 'botoscope_simple_virtual_downloadable';
            protected $virtual = 'yes';
            protected $downloadable = 'yes';

            public function __construct($product = 0) {
                parent::__construct($product);
                $this->product_type = 'botoscope_simple_virtual_downloadable';
                $this->virtual = 'yes';
                $this->downloadable = 'yes';
            }

            public function get_type() {
                return 'botoscope_simple_virtual_downloadable';
            }

            public function is_virtual() {
                return true;
            }

            public function is_downloadable() {
                return true;
            }
        }

        class WC_Product_Botoscope_Simple_Media_Casting extends WC_Product_Simple {

            protected $product_type = 'botoscope_simple_media_casting';

            public function __construct($product = 0) {
                parent::__construct($product);
                $this->product_type = 'botoscope_simple_media_casting';
            }

            public function get_type() {
                return 'botoscope_simple_media_casting';
            }

            public function is_virtual($context = 'view') {
                return true;
            }

            public function is_downloadable($context = 'view') {
                return true;
            }
        }

    }

    add_filter('woocommerce_product_class', function ($classname, $product_type) {
        switch ($product_type) {
            case 'botoscope_simple_virtual':
                $classname = 'WC_Product_Botoscope_Simple_Virtual';
                break;
            case 'botoscope_simple_virtual_downloadable':
                $classname = 'WC_Product_Botoscope_Simple_Virtual_Downloadable';
                break;
            case 'botoscope_simple_media_casting':
                $classname = 'WC_Product_Botoscope_Simple_Media_Casting';
                break;
        }

        return $classname;
    }, 10, 2);
}
